

#!/usr/bin/env bash
set -euo pipefail

# -----------------------------
# Логування
# -----------------------------
log(){  printf "\033[1;32m[OK]\033[0m %s\n" "$*"; }
info(){ printf "\033[1;34m[INFO]\033[0m %s\n" "$*"; }
warn(){ printf "\033[1;33m[WARN]\033[0m %s\n" "$*"; }
err(){  printf "\033[1;31m[ERR]\033[0m %s\n" "$*"; }

need_sudo() {
  if [ "${EUID:-$(id -u)}" -ne 0 ]; then
    warn "Деякі кроки потребують sudo. Вас попросять ввести пароль."
  fi
}

# -----------------------------
# Визначення дистрибутива
# -----------------------------
OS="$(uname -s)"
PM=""

case "$OS" in
  Linux)
    if command -v apt-get >/dev/null 2>&1; then
      PM="apt"
    elif command -v dnf >/dev/null 2>&1; then
      PM="dnf"
    elif command -v pacman >/dev/null 2>&1; then
      PM="pacman"
    else
      err "Unsupported Linux distribution"
    fi
    ;;
  *)
    err "Unsupported OS: $OS"
    ;;
esac

need_sudo

# -----------------------------
# Оновлення індексів
# -----------------------------
if [ "$PM" = "apt" ]; then
  sudo apt-get update -y || warn "APT update failed"
fi

# -----------------------------
# Базові пакети
# -----------------------------
if [ "$PM" = "apt" ]; then
  sudo apt-get install -y \
    build-essential pkg-config libpq-dev default-libmysqlclient-dev \
    ca-certificates gnupg software-properties-common curl || true
fi

# -----------------------------
# Знаходимо Python
# -----------------------------
PYBIN=""
for cand in python3.12 python3 python; do
  if command -v "$cand" >/dev/null 2>&1; then
    PYBIN="$(command -v "$cand")"
    break
  fi
done

if [ -z "$PYBIN" ]; then
  info "Python3 не знайдено — встановлюю..."
  if [ "$PM" = "apt" ]; then
    sudo apt-get install -y python3
  fi
fi

for cand in python3.12 python3 python; do
  if command -v "$cand" >/dev/null 2>&1; then
    PYBIN="$(command -v "$cand")"
  fi
done

if [ -z "$PYBIN" ]; then
  err "Python3 не знайдено навіть після інсталяції."
fi

log "Використовую інтерпретатор: $("$PYBIN" -V) ($PYBIN)"

# -----------------------------
# ensurepip → повністю пропускаємо  
# pip у системі НЕ використовуємо (PEP 668)
# -----------------------------
ensure_pip() {
  log "Пропускаю встановлення системного pip (PEP 668)"
}
ensure_pip

# -----------------------------
# Встановлюємо virtualenv через APT
# -----------------------------
ensure_virtualenv() {
  if command -v virtualenv >/dev/null 2>&1; then
    log "virtualenv вже встановлено"
    return 0
  fi

  info "Встановлюю virtualenv через apt (PEP 668-safe)"
  if [ "$PM" = "apt" ]; then
    sudo apt-get install -y python3-virtualenv
  else
    err "На цьому дистрибутиві неможливо безпечно встановити virtualenv"
  fi

  if ! command -v virtualenv >/dev/null 2>&1; then
    err "virtualenv не знайдено після встановлення"
  fi
}

ensure_virtualenv

# -----------------------------
# Створення / пересоздання venv
# -----------------------------
VENV_DIR="${VENV_DIR:-venv}"

info "Перевіряю virtualenv: $VENV_DIR"

if [ -d "$VENV_DIR" ]; then
  if [ ! -x "$VENV_DIR/bin/pip" ]; then
    warn "Venv пошкоджений — видаляю"
    rm -rf "$VENV_DIR"
  fi
fi

if [ ! -d "$VENV_DIR" ]; then
  info "Створюю venv через virtualenv"
  virtualenv "$VENV_DIR"
else
  log "Venv вже існує"
fi

# pip у venv — дозволений!
"$VENV_DIR/bin/python" -m pip install --upgrade pip setuptools wheel || true

log "Venv готовий: $VENV_DIR"
log "check_python.sh завершено."
