#!/usr/bin/env bash
set -euo pipefail

# -----------------------------
# Helpers
# -----------------------------
install_package_if_missing() {
    local package_name=$1
    if ! dpkg -l | awk '{print $1,$2}' | grep -q "^ii ${package_name}$"; then
        echo "[APT] ${package_name} is not installed. Installing..."
        sudo apt-get install -y "${package_name}"
        echo "[APT] ${package_name} has been installed."
    else
        echo "[APT] ${package_name} is already installed."
    fi
}

have_pkg() {
    local package_name=$1
    apt-cache policy "${package_name}" 2>/dev/null | grep -q 'Candidate:'
}

# -----------------------------
# Update APT once
# -----------------------------
echo "[APT] Updating package lists..."
sudo apt-get update -y || true

# -----------------------------
# Base tools for repos & builds
# -----------------------------
install_package_if_missing "software-properties-common"  # for add-apt-repository
install_package_if_missing "ca-certificates"
install_package_if_missing "gnupg"
install_package_if_missing "pkg-config"
install_package_if_missing "build-essential"

# -----------------------------
# Ensure Python 3.12 availability
# -----------------------------
if ! command -v python3.12 >/dev/null 2>&1; then
    echo "[PY312] python3.12 not found, checking repos..."

    if have_pkg "python3.12"; then
        echo "[PY312] python3.12 available in current repos."
    else
        # Try enabling Deadsnakes PPA on Ubuntu
        if command -v lsb_release >/dev/null 2>&1 && [[ "$(lsb_release -is 2>/dev/null || echo '')" == "Ubuntu" ]]; then
            echo "[PY312] Adding Deadsnakes PPA for Python 3.12..."
            sudo add-apt-repository -y ppa:deadsnakes/ppa
            sudo apt-get update -y || true
        else
            echo "[WARN] Could not find python3.12 in repos and OS is not Ubuntu (or lsb_release missing)."
            echo "       Please ensure a repo with Python 3.12 is available for your distro."
        fi
    fi
fi

# Install Python 3.12 runtime + headers + venv
install_package_if_missing "python3.12"
# Dev headers (for building mysqlclient, etc.)
if have_pkg "python3.12-dev"; then
    install_package_if_missing "python3.12-dev"
else
    echo "[PY312] python3.12-dev not found in repos; installing generic python3-dev as a fallback."
    install_package_if_missing "python3-dev"
fi
# venv for 3.12
if have_pkg "python3.12-venv"; then
    install_package_if_missing "python3.12-venv"
else
    echo "[PY312] python3.12-venv not found; installing python3-venv as a fallback."
    install_package_if_missing "python3-venv"
fi

# Ensure pip for Python 3.12
if ! python3.12 -m pip --version >/dev/null 2>&1; then
    echo "[PY312] Bootstrapping pip for python3.12 via ensurepip..."
    python3.12 -m ensurepip --upgrade || true
fi
python3.12 -m pip install --upgrade pip setuptools wheel || true

# Keep default python3/pip3 present & updated (не замінюємо системний python3)
install_package_if_missing "python3"
install_package_if_missing "python3-pip"
sudo -H pip3 install --upgrade pip || true

# -----------------------------
# DB client headers
# -----------------------------
# Prefer default-libmysqlclient-dev, fallback to libmysqlclient-dev
if have_pkg "default-libmysqlclient-dev"; then
    install_package_if_missing "default-libmysqlclient-dev"
else
    echo "[MYSQL] default-libmysqlclient-dev not found. Falling back to libmysqlclient-dev."
    install_package_if_missing "libmysqlclient-dev"
fi

# PostgreSQL headers (як у вас)
install_package_if_missing "libpq-dev"

# -----------------------------
# Gunicorn
# -----------------------------
install_package_if_missing "gunicorn"

# Also install/upgrade gunicorn in active venv (якщо він активний)
if [[ -n "${VIRTUAL_ENV-}" ]]; then
    echo "[PIP] Detected active virtualenv: ${VIRTUAL_ENV}"
    "${VIRTUAL_ENV}/bin/python" -m pip install --upgrade pip
    "${VIRTUAL_ENV}/bin/python" -m pip install --upgrade gunicorn
else
    echo "[INFO] No active virtualenv detected. Skipping pip install of gunicorn inside venv."
    echo "       (Activate your venv and re-run this script if you want gunicorn inside it.)"
fi

# -----------------------------
# Tips for mysqlclient build with 3.12 (optional)
# -----------------------------
echo "[INFO] To build mysqlclient for Python 3.12 inside your venv, use:"
echo "       /path/to/venv/bin/python -m pip install --no-cache-dir mysqlclient"

echo "[DONE] Environment ready: Python 3.12 (+dev, +venv), pip, build tools, MySQL/Postgres headers, pkg-config, and gunicorn."

