from datetime import datetime, timedelta
from concurrent.futures import ThreadPoolExecutor
import uuid
import mimetypes
import base64
import logging
import os
import sys
import json
import smtplib

from flask_mail import Mail, Message
from flask import render_template_string, has_request_context, flash
from jinja2 import TemplateSyntaxError, UndefinedError
import traceback

from k2.k2cfg import K2
from k2.k2obj import K2Obj
try:
    from components.k2site.k2site.models import K2Emails
except Exception as ex:
    logging.error(f"{ex} K2Site is missing for correct work")

from app import app
db = K2.db
time_zone = K2.timezone

# ─────────────────────────────────────────────────────────────────────────────
# ПУЛ ПОТОКІВ ДЛЯ НЕБЛОКУЮЧОЇ ВІДПРАВКИ
# ─────────────────────────────────────────────────────────────────────────────
MAIL_EXECUTOR = ThreadPoolExecutor(max_workers=4)
MAIL_JOBS = {}  # job_id -> Future

# ─────────────────────────────────────────────────────────────────────────────
# КОНФІГ ПОШТИ (З JSON) + ІНІЦІАЛІЗАЦІЯ Flask-Mail
# ─────────────────────────────────────────────────────────────────────────────
class K2Mail(K2Obj):
    name='k2mail'

    # Конфіги лімітів вкладень (можна змінити в app.config до створення екземпляра)
    app.config.setdefault("MAIL_MAX_ATTACHMENT_MB", 10)       # максимум на 1 файл
    app.config.setdefault("MAIL_MAX_TOTAL_ATTACHMENT_MB", 20) # сумарно

    # шлях до налаштувань
    settings_file_path = os.path.join(
        os.path.dirname(__file__), '..', '..', 'cfg', K2.proj_config, 'settings', 'mail_settings.json'
    )

    def __init__(self, *args, **kwargs):
        super().__init__()

    # ─────────────────────────────────────────────────────────────────────────
    # Допоміжні методи для вкладень
    # ─────────────────────────────────────────────────────────────────────────
    @staticmethod
    def _read_file_safe(path: str) -> bytes:
        # базова нормалізація; за потреби обмеж базову директорію проєкту
        norm = os.path.normpath(path)
        with open(norm, "rb") as f:
            return f.read()

    @staticmethod
    def _guess_mime(filename: str) -> str:
        ctype, _ = mimetypes.guess_type(filename)
        return ctype or "application/octet-stream"

    def _normalize_attachments(self, attachments):
        normalized = []
        if not attachments:
            return normalized

        if not isinstance(attachments, (list, tuple)):
            attachments = [attachments]

        for item in attachments:
            try:
                # 1) tuple ("name.ext","mime/type",b"bytes") — вже працює як є
                if isinstance(item, tuple) and len(item) == 3:
                    filename, content_type, data = item
                    normalized.append({"filename": filename, "content_type": content_type, "data": data})
                    continue

                # 2) dict із шляхом, красивим ім’ям і опційним content_type
                if isinstance(item, dict) and "path" in item:
                    path = item["path"]
                    filename = item.get("filename") or os.path.basename(path)
                    content_type = item.get("content_type") or (
                                mimetypes.guess_type(filename)[0] or "application/octet-stream")
                    with open(path, "rb") as f:
                        data = f.read()
                    normalized.append({"filename": filename, "content_type": content_type, "data": data})
                    continue

                # 3) простий рядок-шлях — теж підтримати
                if isinstance(item, str):
                    filename = os.path.basename(item)
                    content_type = mimetypes.guess_type(filename)[0] or "application/octet-stream"
                    with open(item, "rb") as f:
                        data = f.read()
                    normalized.append({"filename": filename, "content_type": content_type, "data": data})
                    continue

                # інші формати — як було (data_b64, data, Path тощо) якщо ти вже додавав підтримку
                ...
            except Exception as e:
                logging.error(f"Skip attachment due to error: {e}")

        return normalized

    def _enforce_attachment_limits(self, items):
        """Застосовує ліміти на розмір (за конфігом)."""
        max_each = int(app.config.get("MAIL_MAX_ATTACHMENT_MB", 10)) * 1024 * 1024
        max_total = int(app.config.get("MAIL_MAX_TOTAL_ATTACHMENT_MB", 20)) * 1024 * 1024

        total = 0
        kept = []
        for it in items:
            size = len(it["data"])
            if size > max_each:
                logging.warning(f"Attachment '{it['filename']}' skipped (>{max_each} bytes)")
                K2.logging_message(K2.log_error, "Розмір файлу перевищено, файл не відправлено")
                continue
            if total + size > max_total:
                logging.warning(f"Total attachments limit exceeded, skipping '{it['filename']}'")
                K2.logging_message(K2.log_error, "Розмір файлів перевищено, файли не відправлено")
                continue
            kept.append(it)
            total += size
        return kept

    # ─────────────────────────────────────────────────────────────────────────
    # ПІДГОТОВКА ТІЛА ЛИСТА (шаблон або custom_body)
    # ─────────────────────────────────────────────────────────────────────────
    def _looks_like_inline_template(self, s: str) -> bool:
        s = (s or "").strip()
        if not s:
            return False
        # евристика: якщо є jinja-теги або html-теги — вважаємо це inline-шаблон
        return ("{{" in s) or ("{%" in s) or ("</" in s) or ("<html" in s.lower())

    def _render_jinja_string(self, template_str: str, **ctx) -> str | None:
        try:
            t = app.jinja_env.from_string(template_str)  # використовує той самий Jinja env, але без context processors
            return t.render(**ctx)
        except (TemplateSyntaxError, UndefinedError) as e:
            logging.exception(f"Jinja render error: {e}")
            return None

    def _prepare_email_body(self, subject, email, email_template, redirect_url, login, custom_body, **kwargs):
        """
        Повертає HTML або None. НЕ викликає Flask context processors.
        """
        try:
            if custom_body:
                return custom_body

            if email_template:
                # 1) inline-шаблон
                s = (email_template or "").strip()
                looks_inline = ("{{" in s) or ("{%" in s) or ("</" in s) or ("<html" in s.lower())
                if looks_inline and not os.path.exists(email_template):
                    return self._render_jinja_string(
                        s,
                        user_email=email,
                        login=login,
                        reset_url=redirect_url,
                        domain=K2.domain,
                        **kwargs
                    )

                # 2) файл-шаблон
                try:
                    with app.open_resource(email_template) as f:
                        tpl = f.read().decode("utf-8")
                except Exception as fe:
                    logging.exception(f"Email template open failed: {email_template}")
                    return None

                return self._render_jinja_string(
                    tpl,
                    user_email=email,
                    login=login,
                    reset_url=redirect_url,
                    domain=K2.domain,
                    **kwargs
                )

            raise ValueError("Either 'custom_body' or 'email_template' must be provided.")
        except Exception as e:
            logging.exception(f"Error preparing email content: {e}")
            # не використовуємо flash тут, бо це може бути поза request
            return None
    # ─────────────────────────────────────────────────────────────────────────
    # ВНУТРІШНІЙ ВИКОНАВЕЦЬ (СИНХРОННО): РЕАЛЬНА ВІДПРАВКА + ЛОГ В БД
    # ─────────────────────────────────────────────────────────────────────────
    def _send_email_impl(self, subject, email, email_template=None, redirect_url=None,
                         login=None, custom_body=None, attachments=None, **kwargs):
        """
        Виконується у потоці або синхронно.
        """
        with app.app_context():
            email_body = self._prepare_email_body(subject, email, email_template, redirect_url, login, custom_body, **kwargs)
            if not email_body:
                return False

            try:
                msg = Message(subject, recipients=[email], html=email_body)

                # вкладення
                items = self._normalize_attachments(attachments)
                items = self._enforce_attachment_limits(items)
                for it in items:
                    try:
                        msg.attach(
                            filename=it["filename"],
                            content_type=it["content_type"],
                            data=it["data"]
                        )
                    except Exception as aex:
                        logging.error(f"Failed to attach '{it['filename']}': {aex}")

                mail.send(msg)

                # логування у БД (за можливості)
                try:
                    new_email = K2Emails(
                        emailid=K2.generate_id(),
                        from_email=app.config.get('MAIL_DEFAULT_SENDER'),
                        to_email=email,
                        message_title=subject,
                        message=email_body,
                        active=1,
                        createdate=datetime.now(time_zone),
                        updatedate=datetime.now(time_zone),
                    )
                    try:
                        filenames = [it["filename"] for it in items]
                        if hasattr(new_email, "message_files"):
                            new_email.message_files = json.dumps(filenames, ensure_ascii=False)
                    except Exception:
                        pass

                    db.session.add(new_email)
                    db.session.commit()
                except Exception as e:
                    logging.error(f'Error saving mail history to DB: {str(e)}')

                return True

            except Exception as e:
                logging.error(f'Error sending mail: {str(e)}')
                if has_request_context():
                    try:
                        flash(f'Error sending mail: {e}', 'danger')
                    except Exception:
                        pass
                return False

    # ─────────────────────────────────────────────────────────────────────────
    # ПУБЛІЧНИЙ API: СИНХРОННО / АСИНХРОННО
    # ─────────────────────────────────────────────────────────────────────────
    def send_email(self, subject, email, email_template=None, redirect_url=None,
                   login=None, custom_body=None, attachments=None, async_=False, **kwargs):
        """
        Якщо async_=True — ставить відправку у пул потоків і повертає одразу {"status":"queued","job_id":...}.
        Якщо async_=False — працює синхронно (як раніше), повертає True/False.
        """
        if not async_:
            return self._send_email_impl(
                subject, email, email_template, redirect_url,
                login, custom_body, attachments, **kwargs
            )

        job_id = uuid.uuid4().hex
        fut = MAIL_EXECUTOR.submit(
            self._send_email_impl,
            subject, email, email_template, redirect_url,
            login, custom_body, attachments, **kwargs
        )
        MAIL_JOBS[job_id] = fut
        return {"status": "queued", "job_id": job_id}

    @staticmethod
    def get_mail_job_status(job_id: str):
        fut = MAIL_JOBS.get(job_id)
        if not fut:
            return {"status": "unknown"}
        if fut.done():
            err = fut.exception()
            return {"status": "done" if err is None else "failed", "error": str(err) if err else None}
        return {"status": "running"}

    # ─────────────────────────────────────────────────────────────────────────
    # ЗБЕРЕЖЕННЯ / ЗАВАНТАЖЕННЯ НАЛАШТУВАНЬ SMTP
    # ─────────────────────────────────────────────────────────────────────────
    @classmethod
    def save_mail_settings(cls, settings_data):
        """Функція для збереження налаштувань поштового сервера"""
        try:
            os.makedirs(os.path.dirname(cls.settings_file_path), exist_ok=True)

            if cls.test_mail_connection(settings_data):
                with open(cls.settings_file_path, 'w') as f:
                    json.dump(settings_data, f, indent=4)
                from .k2upd import reload_app
                reload_app()
                return True
            else:
                logging.error("Invalid mail settings, not saving.")
                return False
        except Exception as e:
            logging.error(f"Error saving settings: {e}")
            return False

    @classmethod
    def load_settings(cls):
        """Завантажує налаштування з JSON файлу."""
        try:
            if os.path.exists(cls.settings_file_path):
                with open(cls.settings_file_path, 'r') as f:
                    return json.load(f)
            return {}
        except Exception as e:
            logging.error(e)
            return {}

    @classmethod
    def test_mail_connection(cls, settings_data):
        """Перевірка підключення до SMTP."""
        try:
            if settings_data.get('mail_encryption') == 'ssl':
                server = smtplib.SMTP_SSL(settings_data['mail_server'], settings_data['mail_port'])
            else:
                server = smtplib.SMTP(settings_data['mail_server'], settings_data['mail_port'])
                if settings_data.get('mail_encryption') == 'tls':
                    server.starttls()

            server.login(settings_data['mail_username'], settings_data['mail_password'])
            server.quit()
            return True
        except (smtplib.SMTPException, Exception) as e:
            logging.error(f"SMTP connection failed: {e}")
            return False


# ─────────────────────────────────────────────────────────────────────────────
# ІНІЦІАЛІЗАЦІЯ ПОШТОВИХ НАЛАШТУВАНЬ ТА MAIL(app)
# ─────────────────────────────────────────────────────────────────────────────
mail_settings = K2Mail.load_settings()
app.config['MAIL_SERVER'] = mail_settings.get('mail_server')
app.config['MAIL_PORT'] = mail_settings.get('mail_port')

if mail_settings.get('mail_encryption', 'ssl') == 'ssl':
    app.config['MAIL_USE_SSL'] = True
else:
    app.config['MAIL_USE_SSL'] = False

if mail_settings.get('mail_encryption', 'tls') == 'tls':
    app.config['MAIL_USE_TLS'] = True
else:
    app.config['MAIL_USE_TLS'] = False

app.config['MAIL_USERNAME'] = mail_settings.get('mail_username')
app.config['MAIL_PASSWORD'] = mail_settings.get('mail_password')
app.config['MAIL_DEFAULT_SENDER'] = mail_settings.get('mail_default_sender')

mail = Mail(app)
