import uuid

from flask_login import current_user
from flask import jsonify, send_from_directory, url_for
from requests import session
# from markdown_it.common.html_re import comment
from sqlalchemy.orm import sessionmaker
from datetime import datetime, timedelta
from k2.k2obj import K2Obj
from sqlalchemy.testing.plugin.plugin_base import logging
from sqlalchemy.orm import aliased
from sqlalchemy import or_, and_, desc, cast, Date
import logging


class K2Notifications(K2Obj):
    """Клас створення та відправки сповіщень в системі"""
    name = 'k2notifications'

    def __init__(self, *args, **kwargs):
        super().__init__()

    @classmethod
    def __generate_id(cls):
        '''Генерація ID'''
        generated_id = uuid.uuid4().hex[:32]
        return ''.join([generated_id[i:i + 4] for i in range(0, len(generated_id), 4)])

    @classmethod
    def save_approve_obj(cls,
                         obj_id,
                         object_row_id=None,
                         parentid='0',
                         user_list=[],
                         url=None,
                         description=None,
                         createuser=None,
                         system=None,
                         comment='',
                         show_btn=1
                         ):
        """
            Зберігає або оновлює запити на затвердження для конкретного об'єкта.

            Цей метод обробляє створення або оновлення запитів на затвердження.
            Для заданого об'єкта та списку користувачів він забезпечує, що кожен користувач отримає
            відповідне сповіщення, і оновлює його у разі, якщо запис вже існує.

            Параметри:
                obj_id (str): Ідентифікатор об'єкта, для якого створюються запити на затвердження.
                object_row_id (str, optional): Додатковий ідентифікатор рядка об'єкта. За замовчуванням None.
                parentid (str, optional): Ідентифікатор батьківського об'єкта. За замовчуванням '0'.
                user_list (list): Список ідентифікаторів користувачів, яким будуть надіслані сповіщення.
                url (str): URL для процесу затвердження.
                description (str, optional): Опис запиту на затвердження. За замовчуванням None.
                createuser (str, optional): Користувач, який створює запит. За замовчуванням None.
                system (bool, optional): Прапорець, що вказує, чи є запит системним. За замовчуванням None.
                comment (str, optional): Коментар до запиту. За замовчуванням ''.
                show_btn (int, optional): Прапорець для відображення кнопки. За замовчуванням 1.

            Повертає:
                Response: JSON-відповідь із статусом успіху або помилки.
            """
        from k2.k2cfg import K2
        from components.k2site.k2site.models import K2Approval, K2users, K2Usernotification
        from app import socketio, db
        if not obj_id or not url or not user_list:
            return jsonify({"error": "Object ID, URL or user list is missing"}), 400
        # Session = sessionmaker(bind=cls.db)
        # session = Session()
        try:
            if system:
                system = 1
            else:
                system = 0
        except Exception as e:
            K2.logging_message(status=K2.log_error, message=f"{e}")
        try:
            # Проходимо по кожному користувачеві і створюємо або оновлюємо запис у таблиці K2Approval
            for user_id in user_list:
                # Перевіряємо чи існує запис з таким object_id і user_id
                existing_approval = db.session.query(K2Usernotification).filter_by(object_id=obj_id,
                                                                                user_id=user_id,
                                                                                parentid='0'
                                                                                ).first()
                notification_id = cls.__generate_id()  # Генеруємо унікальний notification_id
                if not createuser:
                    createuser=current_user.get_id(),
                if not existing_approval:
                # Якщо запис не існує, створюємо новий
                    new_approval = K2Usernotification(
                        notification_id=notification_id,
                        object_id=obj_id,
                        object_row_id=object_row_id,
                        user_id=user_id,
                        is_approved=-1,  # Спочатку статус не затверджено
                        description=description,
                        approval_url=url,
                        createuser=createuser,
                        updateuser=createuser,
                        createdate=datetime.now(),
                        updatedate=datetime.now(),
                        parentid=parentid,
                        system=system,
                        show_btn=show_btn,
                        is_read=False,
                        active=1
                    )
                    db.session.add(new_approval)
                else:
                    existing_approval.is_approved = -1
                    existing_approval.approval_url = url
                    existing_approval.updatedate = datetime.now()
                    existing_approval.comment = ''
                    existing_approval.description = description
                    existing_approval.createuser = createuser
                    existing_approval.is_read = False

                try:
                    db.session.commit()
                except Exception as commit_error:
                    db.session.rollback()
                    return jsonify({"error": str(commit_error)}), 500
            socketio.emit('update-notifications-count')

            return jsonify({"status": "success", "message": "Approval requests created/updated"}), 200
        except Exception as e:
            K2.logging_message(status=K2.log_error, message=f"{e}")
            return jsonify({"error": str(e)}), 500
        # finally:
        #     session.close()


    @classmethod
    def save_approve_obj_return(cls,
                                notification_id,
                                obj_id,
                                object_row_id=None,
                                is_approved=1,
                                url=None,
                                description=None,
                                createuser=None,
                                comment='',
                                send_email=True):
        """
                    Зберігає або оновлює зворотній запит на затвердження для конкретного об'єкта.

                    Цей метод обробляє створення або оновлення зворотніх запитів на затвердження`.
                    Для заданого об'єкта та списку користувачів він забезпечує, що кожен користувач отримає
                    відповідне сповіщення, і оновлює його у разі, якщо запис вже існує.

                    Параметри:
                        obj_id (str): Ідентифікатор об'єкта, для якого створюються запити на затвердження.
                        object_row_id (str, optional): Додатковий ідентифікатор рядка об'єкта. За замовчуванням None.
                        parentid (str, optional): Ідентифікатор батьківського об'єкта. За замовчуванням '0'.
                        user_list (list): Список ідентифікаторів користувачів, яким будуть надіслані сповіщення.
                        url (str): URL для процесу затвердження.
                        description (str, optional): Опис запиту на затвердження. За замовчуванням None.
                        createuser (str, optional): Користувач, який створює запит. За замовчуванням None.
                        system (bool, optional): Прапорець, що вказує, чи є запит системним. За замовчуванням None.
                        comment (str, optional): Коментар до запиту. За замовчуванням ''.
                        show_btn (int, optional): Прапорець для відображення кнопки. За замовчуванням 1.

                    Повертає:
                        Response: JSON-відповідь із статусом успіху або помилки.
                    """
        from app import K2
        from components.k2site.k2site.models import K2Approval, K2users, K2Usernotification
        from app import socketio, db
        # from sqlalchemy.orm import sessionmaker
        from datetime import datetime

        if not obj_id:
            return jsonify({"error": "Object ID, URL or user list is missing"}), 400

        # Session = sessionmaker(bind=cls.db)
        # session = Session()

        try:
            # Знайдемо запис батьківського документа (де parentid = 0)
            parent_record = db.session.query(K2Usernotification).filter_by(object_id=obj_id, parentid='0', notification_id=notification_id).first()

            if parent_record:
                # Перевірити, чи існує запис з ненульовим parentid
                existing_approval_record = (db.session.query(K2Usernotification)
                                            .filter(K2Usernotification.object_id == obj_id,
                                                    K2Usernotification.is_approved == '0',
                                                    K2Usernotification.user_id == parent_record.createuser,
                                                    K2Usernotification.parentid != '0',
                                                    K2Usernotification.notification_id == notification_id)
                                            .first())

                if not existing_approval_record:
                    try:
                        if not createuser:
                            createuser = current_user.get_id()
                        # Створюємо новий запис із parentid, який дорівнює notification_id батьківського запису
                        new_approval_record = K2Usernotification(
                            notification_id=K2.generate_id(),
                            object_id=obj_id,
                            user_id=parent_record.createuser,
                            object_row_id=parent_record.object_row_id,
                            parentid=notification_id,  # Використовуємо notification_id як parentid для нового запису
                            is_approved=is_approved,
                            description=parent_record.description,
                            approval_url=parent_record.approval_url,
                            createdate=datetime.now(),
                            updatedate=datetime.now(),
                            createuser=createuser,  # Використовуємо createuser батьківського запису
                            updateuser=createuser,
                            comment=comment,
                            show_btn=parent_record.show_btn,
                            is_read = False,
                        )
                        db.session.add(new_approval_record)
                    except Exception as e:
                        K2.logging_message(status=K2.log_error, message=f"{e}")
                else:
                    existing_approval_record.is_approved = is_approved
                    existing_approval_record.updatedate = datetime.now()
                    existing_approval_record.updateuser = parent_record.user_id
                    existing_approval_record.comment = comment
                    existing_approval_record.is_read = False,
                # Оновлюємо статус батьківського запису, якщо документ затверджений
                if is_approved == 1:
                    parent_record.is_approved = 1
                    parent_record.updatedate = datetime.now()
                    parent_record.is_read = False
                else:
                    parent_record.is_approved = 0
                    parent_record.comment = comment
                    parent_record.updatedate = datetime.now()
                    parent_record.is_read = False
                db.session.commit()
                if send_email:
                    try:
                        # Відправляємо сповіщення на email
                        subject = f'Сповіщення користувача - {parent_record.description}'
                        email_template = 'components/k2site/k2site/templates/email/user_notifications.html'
                        # Отримуємо користувачів зі списку для погодження
                        from .k2mail import K2Mail
                        user =  db.session.query(K2users).filter(K2users.user_id == parent_record.createuser, K2users.active == 1).first()
                        if user.email:
                            K2Mail().send_email(subject, user.email, email_template, parent_record.approval_url)
                    except Exception as e:
                        logging.error(f"{e}")
                socketio.emit('update-notifications-count')
                return jsonify({"status": "success", "message": "Новий запис створено і статус оновлено!"}), 200
            else:
                # Якщо батьківський запис не знайдено
                return jsonify({"error": "Батьківський документ не знайдено."}), 400

        except Exception as e:
            # Якщо сталася помилка, скасовуємо транзакцію та фіксуємо помилку
            db.session.rollback()
            K2.logging_message(status=K2.log_error, message=f"{e}")
            return jsonify({"error": str(e)}), 500
        #
        # finally:
        #     session.close()


    @classmethod
    def send_approval_request(cls,  user_list, obj_id,  object_row_id=None, parentid='0', url=None, description=None, comment=None, show_btn=1, send_email=True, system=False):
        """
            Відправляє запит на затвердження документа в системі та опціонально надсилає сповіщення електронною поштою.

            Ця функція забезпечує створення або оновлення запитів на затвердження документа для зазначеного списку користувачів
            та надсилає сповіщення в системі.
            Також вона опціонально надсилає сповіщення електронною поштою кожному користувачеві зі списку.

            Параметри:
                user_list (list): Список ідентифікаторів користувачів, які отримають запит на затвердження.
                obj_id (str): Ідентифікатор документа, для якого створюється запит.
                object_row_id (str, optional): Додатковий ідентифікатор рядка документа. За замовчуванням None.
                parentid (str, optional): Ідентифікатор батьківського документа. За замовчуванням '0'.
                url (str, optional): URL, який вказує на документ або процес затвердження. За замовчуванням None.
                description (str, optional): Опис документа або запиту. За замовчуванням None.
                comment (str, optional): Коментар до запиту на затвердження. За замовчуванням None.
                show_btn (int, optional): Прапорець для відображення кнопки у сповіщенні. За замовчуванням 1.
                send_email (bool, optional): Чи надсилати сповіщення електронною поштою. За замовчуванням True.
                system (bool, optional): Прапорець, що вказує, чи є запит системним. За замовчуванням False.

            Повертає:
                Response: JSON-відповідь зі статусом успіху або помилки.

            Побічні ефекти:
                - Зберігає запити на затвердження у базу даних через функцію `save_approve_obj`.
                - Надсилає сповіщення електронною поштою за допомогою класу `K2Mail`.

            Обробка помилок:
                - У разі помилки збереження або відправлення сповіщення транзакція відкочується, і повертається JSON із повідомленням про помилку.
            """
        from components.k2site.k2site.models import K2users
        from app import db
        try:
            # Спочатку зберігаємо запити на затвердження
            cls.save_approve_obj(obj_id=obj_id,
                                 object_row_id=object_row_id,
                                 parentid=parentid,
                                 user_list=user_list,
                                 url=url,
                                 description=description,
                                 system=system,
                                 comment=comment,
                                 show_btn=show_btn)
            #Відправляємо сповіщення на email
            if send_email:
                try:
                    subject = f'Сповіщення користувача - {description}'
                    email_template = 'components/k2site/k2site/templates/email/user_notifications.html'
                    # Отримуємо користувачів зі списку для погодження
                    from .k2mail import K2Mail
                    users = K2users.query.filter(K2users.user_id.in_(user_list), K2users.active == 1).all()
                    for user in users:
                        if user.email:
                            K2Mail().send_email(subject, user.email, email_template, url)
                except Exception as e:
                    logging.error('Error send e-mail' + str(e))

        except Exception as e:
            db.session.rollback()  # Відкочуємо транзакцію в разі помилки
            return jsonify({"error": str(e)}), 500
        finally:
            db.session.close()


    @classmethod
    def send_system_request(cls,  user_list, obj_id, object_row_id=None, parentid='0', url=None, description=None, comment=None, show_btn=1, send_email=True):
        """
           Відправляє системні  сповіщення користувачу.


           Ця функція використовується для автоматизованого створення запитів на затвердження документа від імені системи.
           Вона зберігає запити на затвердження в базі даних та, за необхідності, надсилає електронні сповіщення користувачам.

           Параметри:
               user_list (list): Список ідентифікаторів користувачів, які отримають запит на затвердження.
               obj_id (str): Ідентифікатор документа, для якого створюється запит.
               object_row_id (str, optional): Додатковий ідентифікатор рядка документа. За замовчуванням None.
               parentid (str, optional): Ідентифікатор батьківського документа. За замовчуванням '0'.
               url (str, optional): URL, який вказує на документ або процес затвердження. За замовчуванням None.
               description (str, optional): Опис документа або запиту. За замовчуванням None.
               comment (str, optional): Коментар до запиту на затвердження. За замовчуванням None.
               show_btn (int, optional): Прапорець для відображення кнопки у сповіщенні. За замовчуванням 1.
               send_email (bool, optional): Чи надсилати сповіщення електронною поштою. За замовчуванням True.

           Повертає:
               Response: JSON-відповідь зі статусом успіху або помилки.

           Побічні ефекти:
               - Зберігає системні запити на затвердження у базу даних через функцію `save_approve_obj`.
               - Надсилає сповіщення електронною поштою за допомогою класу `K2Mail`.

           Обробка помилок:
               - У разі помилки збереження або надсилання сповіщень транзакція відкочується, і повертається JSON із повідомленням про помилку.

           Додатково:
               - Встановлює параметр `system=True`, щоб позначити запит як системний.
               - Використовує шаблон електронної пошти `user_notifications.html` для надсилання повідомлень.
           """
        from components.k2site.k2site.models import K2users
        from app import db
        try:
            # Спочатку зберігаємо запити на затвердження
            cls.save_approve_obj(obj_id=obj_id,
                                 object_row_id=object_row_id,
                                 parentid=parentid,
                                 user_list=user_list,
                                 url=url,
                                 description=description,
                                 system=True,
                                 comment=comment,
                                 show_btn=show_btn)
            # Відправляємо сповіщення на email
            if send_email:
                try:
                    subject = f'Сповіщення користувача - {description}'
                    email_template = 'components/k2site/k2site/templates/email/user_notifications.html'
                    # Отримуємо користувачів зі списку для погодження
                    from .k2mail import K2Mail
                    users = K2users.query.filter(K2users.user_id.in_(user_list), K2users.active == 1).all()
                    for user in users:
                        if user.email:
                            K2Mail().send_email(subject, user.email, email_template, url)
                except Exception as e:
                    logging.error('Error send e-mail' + str(e))
        except Exception as e:
                db.session.rollback()  # Відкочуємо транзакцію в разі помилки
                return jsonify({"error": str(e)}), 500
        finally:
            db.session.close()


    @classmethod
    def get_user_notifications(cls):
        """
           Отримує непідтверджені сповіщення (0 або -1) для поточного користувача.

           Ця функція використовується для витягування активних сповіщень для користувача.

           Повертає:
               Response: JSON-відповідь зі списком сповіщень для користувача.
        """
        notifications = []
        from app import db, app
        if hasattr(app, 'login_manager') and current_user.is_authenticated:
            from components.k2site.k2site.models import K2users, K2Usernotification

            try:
                # Отримуємо ідентифікатор поточного користувача
                current_user_id = current_user.get_id()

                # Створюємо два аліаси для таблиці K2users
                creator_user = aliased(K2users)
                approver_user = aliased(K2users)

                today = datetime.utcnow().date()
                delta_days_ago = today - timedelta(days=3)
                # Отримання необхідних даних З БД

                unapproved_notifications = db.session.query(
                    K2Usernotification.notification_id,
                    K2Usernotification.object_id,
                    K2Usernotification.object_row_id,
                    K2Usernotification.is_approved,
                    K2Usernotification.user_id,
                    K2Usernotification.approval_url,
                    K2Usernotification.description,
                    K2Usernotification.createdate,
                    K2Usernotification.createuser,
                    K2Usernotification.updatedate,
                    K2Usernotification.updateuser,
                    K2Usernotification.parentid,
                    K2Usernotification.system,
                    K2Usernotification.comment,
                    K2Usernotification.show_btn,
                    K2Usernotification.is_read,
                    creator_user.name.label('creator_name'),
                    creator_user.photo.label('creator_photo'),
                    approver_user.name.label('approver_name'),
                    approver_user.photo.label('approver_photo')
                ).outerjoin(
                    creator_user, K2Usernotification.createuser == creator_user.user_id
                ).join(
                    approver_user, K2Usernotification.user_id == approver_user.user_id
                ).filter(
                    K2Usernotification.active == 1
                ).filter(
                    or_(
                        K2Usernotification.is_approved == -1,  # Показуємо всі, якщо is_approved == -1
                        and_(
                            K2Usernotification.is_approved != -1,
                            # Інші статуси
                            cast(K2Usernotification.updatedate, Date) >= delta_days_ago
                        )
                    )
                ).filter(
                    or_(
                        K2Usernotification.user_id == current_user_id,
                        and_(
                            K2Usernotification.createuser == current_user_id,  # Друга умова
                            K2Usernotification.system == 1  # Додаємо перевірку system == 1
                        )
                    )
                ).order_by(
                    desc(K2Usernotification.updatedate)
                ).all()


                # Формуємо список сповіщень у зручний для повернення формат
                notifications = [
                    {
                        "notification_id": notif.notification_id,
                        "user_id": notif.user_id,
                        "object_id": notif.object_id,
                        "object_row_id": notif.object_row_id,
                        "is_approved": notif.is_approved,
                        "parentid": notif.parentid,
                        "approval_url": notif.approval_url,
                        "description": notif.description,
                        "createdate": notif.createdate.strftime('%Y-%m-%d %H:%M:%S'),
                        "updatedate":notif.updatedate.strftime('%Y-%m-%d %H:%M:%S'),
                        "createuser": notif.createuser if notif.createuser else None,
                        "updateuser": notif.updateuser,
                        "creator_name": notif.creator_name,
                        "creator_photo": notif.creator_photo if notif.creator_photo else None ,
                        "approver_name": notif.approver_name,
                        "approver_photo": notif.approver_photo if notif.approver_photo else None,
                        "system": notif.system,
                        "comment": notif.comment,
                        "show_btn": notif.show_btn,
                        "is_read": notif.is_read
                    }
                    for notif in unapproved_notifications
                ]
                return  notifications

            except Exception as e:
                notifications = []
                logging.error(e)
                return notifications
        return None

    @classmethod
    def get_all_notifications(cls, limit=10):
        """
           Отримує непідтверджені сповіщення (0 або -1) для поточного користувача.

           Ця функція використовується для витягування активних сповіщень для користувача.

           Аргументи:
               limit (int): Максимальна кількість сповіщень для повернення (за замовчуванням 10).

           Повертає:
               Response: JSON-відповідь зі списком сповіщень для користувача.
        """
        notifications = []
        from app import db, app
        if hasattr(app, 'login_manager') and current_user.is_authenticated:
            from components.k2site.k2site.models import K2users, K2Usernotification

            try:
                # Отримуємо ідентифікатор поточного користувача
                # Створюємо два аліаси для таблиці K2users
                creator_user = aliased(K2users)
                approver_user = aliased(K2users)

                today = datetime.utcnow().date()
                delta_days_ago = today - timedelta(days=30)

                # Отримання необхідних даних З БД
                unapproved_notifications = db.session.query(
                    K2Usernotification.notification_id,
                    K2Usernotification.object_id,
                    K2Usernotification.object_row_id,
                    K2Usernotification.is_approved,
                    K2Usernotification.user_id,
                    K2Usernotification.approval_url,
                    K2Usernotification.description,
                    K2Usernotification.createdate,
                    K2Usernotification.createuser,
                    K2Usernotification.updatedate,
                    K2Usernotification.updateuser,
                    K2Usernotification.parentid,
                    K2Usernotification.system,
                    K2Usernotification.comment,
                    K2Usernotification.show_btn,
                    K2Usernotification.is_read,
                    creator_user.name.label('creator_name'),
                    creator_user.photo.label('creator_photo'),
                    approver_user.name.label('approver_name'),
                    approver_user.photo.label('approver_photo')
                ).outerjoin(
                    creator_user, K2Usernotification.createuser == creator_user.user_id
                ).join(
                    approver_user, K2Usernotification.user_id == approver_user.user_id
                ).filter(
                    cast(K2Usernotification.updatedate, Date) >= delta_days_ago
                        ).order_by(
                    desc(K2Usernotification.updatedate)
                ).limit(limit).all()  # Додаємо обмеження кількості записів

                # Формуємо список сповіщень у зручний для повернення формат
                notifications = [
                    {
                        "notification_id": notif.notification_id,
                        "user_id": notif.user_id,
                        "object_id": notif.object_id,
                        "object_row_id": notif.object_row_id,
                        "is_approved": notif.is_approved,
                        "parentid": notif.parentid,
                        "approval_url": notif.approval_url,
                        "description": notif.description,
                        "createdate": notif.createdate.strftime('%Y-%m-%d %H:%M:%S'),
                        "updatedate": notif.updatedate.strftime('%Y-%m-%d %H:%M:%S'),
                        "createuser": notif.createuser if notif.createuser else None,
                        "updateuser": notif.updateuser,
                        "creator_name": notif.creator_name,
                        "creator_photo": notif.creator_photo if notif.creator_photo else None,
                        "approver_name": notif.approver_name,
                        "approver_photo": notif.approver_photo if notif.approver_photo else None,
                        "system": notif.system,
                        "comment": notif.comment,
                        "show_btn": notif.show_btn,
                        "is_read": notif.is_read
                    }
                    for notif in unapproved_notifications
                ]
                return notifications

            except Exception as e:
                notifications = []
                logging.error(e)
                return notifications
        return None

    @classmethod
    def mark_as_read(cls, notification_id, user_id):
        """
        Позначає сповіщення як прочитане.

        Параметри:
            notification_id (str): Ідентифікатор сповіщення.
            user_id (str): Ідентифікатор користувача.

        Повертає:
            dict: Статус операції (успішно чи з помилкою).
        """
        from app import db
        notification = db.session.filter_by(notification_id=notification_id, user_id=user_id).first()
        if notification:
            notification.is_read = True
            db.session.commit()
            return {"status": "success", "message": "Messages read"}
        return {"status": "error", "message": "Messages not found"}

    @classmethod
    def approve_all_notifications_by_object(cls, obj_id, updateuser):
        from components.k2site.k2site.models import K2users, K2Usernotification
        from k2.k2cfg import K2
        from app import db
        from sqlalchemy.orm import sessionmaker
        from datetime import datetime
        #
        # Session = sessionmaker(bind=cls.db)
        # session = Session()

        try:
            records = db.session.query(K2Usernotification).filter(
                K2Usernotification.object_id == obj_id,
                K2Usernotification.is_approved != 1
            ).all()

            for record in records:
                record.is_approved = 1
                record.updatedate = datetime.now()
                record.updateuser = updateuser

            db.session.commit()  # ✅ commit після циклу — ефективніше
            return True

        except Exception as e:
            db.session.rollback()
            K2.logging_message(status=K2.log_error, message=f"{e}")
            return False

        # finally:
        #     cls.session.close()
