import logging
import os
from datetime import datetime
import uuid
import re

from flask_login import LoginManager, current_user
from flask import request, send_from_directory
from sqlalchemy.orm import sessionmaker

from k2.k2obj import K2Obj


class K2Path(K2Obj):
    '''клас формування і пошуку ієрархії файлової системи проектів
        та збереження файлів
    '''
    name = 'k2path'
    def __init__(self, *args, **kwargs):
        super().__init__()

    #     self._data_path = None
    #
    # @property
    # def data_path(self):
    #     """Getter"""
    #     return self._data_path
    #
    # @data_path.setter
    # def name_yml(self, name: str):
    #     """Setter method"""
    #     self._data_path = name

    @classmethod
    def __generate_id(cls):
        '''Генерація ID'''
        generated_id = uuid.uuid4().hex[:32]
        return ''.join([generated_id[i:i + 4] for i in range(0, len(generated_id), 4)])

    def __current_proj(self, current_project_id):
        from app import app
        if hasattr(app, 'login_manager') and isinstance(app.login_manager,
                                                        LoginManager) and current_user.is_authenticated:
            from components.k2site.k2site.models import K2Proj
            current_project = K2Proj.query.filter_by(projid=current_project_id).first()
        return current_project


    def __current_usr(self):
        from app import app
        if hasattr(app, 'login_manager') and isinstance(app.login_manager,
                                                        LoginManager) and current_user.is_authenticated:
            from components.k2site.k2site.models import K2Proj, K2users
            user_id = current_user.get_id()
            user = K2users.query.get(user_id)
        return user


    def __current_countreparts(self):
        from app import app
        if hasattr(app, 'login_manager') and isinstance(app.login_manager,
                                                        LoginManager) and current_user.is_authenticated:
            from components.k2site.k2site.models import K2Proj, K2users
            user_id = current_user.get_id()
            user = K2users.query.get(user_id)
            if user is not None:
                current_counterpart_id = user.counterpart_id
            else:
                current_counterpart_id = None
        return current_counterpart_id


    def __current_stor(self):
        from app import app
        if hasattr(app, 'login_manager') and isinstance(app.login_manager,
                                                        LoginManager) and current_user.is_authenticated:
            from components.k2site.k2site.models import K2Proj, K2users
            user_id = current_user.get_id()
            user = K2users.query.get(user_id)
            if user is not None:
                current_storage_id = user.storage_id
            else:
                current_storage_id = None
        return current_storage_id


    def __traverse_hierarchy(self, project, project_names):
        """Recursive function to traverse the project hierarchy.

           Args:
               project: The current project object.
               project_names (list): A list to store the project IDs as they are traversed.

           Returns:
               None

           Note:
               This function recursively traverses the project hierarchy starting from the given project
               and collects the project IDs in the provided list. It traverses upwards in the hierarchy
               until it reaches the root project.

           """
        #nonlocal project_names
        parent_project = self.__current_proj(project.parentid)
        if parent_project:
            project_names.append(parent_project.projid)
            # If there is a parent project, recursively call the function for it
            self.__traverse_hierarchy(parent_project, project_names)


    def __find_path(self, current_project=None):
        '''Recursive function to determine the path to the folder
           of the user in the project hierarchy.

           Usage: K2().find_path()

           Returns: Folder hierarchy for projects for current user
        '''
        user = self.__current_usr()
        if user:
            if not current_project:
                current_project_id = user.projid  #отримали поточний проект
                current_project = self.__current_proj(current_project_id)
                project_names = [current_project.projid] if current_project else []  # Початковий список з іменем(id) поточного проекту
            else:
                project_names = [current_project.projid]
        # Викликати рекурсивну функцію
        self.__traverse_hierarchy(current_project, project_names)

        # Сформувати шлях, об'єднавши імена проектів від кінця до початку
        user_project_path = '/'.join(reversed(project_names))
        return user_project_path


    def data_path(self, work_path=None, user_folder=False):
        '''
        Create folder hierarchy and return path

        Args:
        work_path - str (end work_path)
        user_folder - boolean (False: defaulL , True: if need create user folder)

        Returns: str user data folder path hierarchy

        Usage: K2().path.data_path('conf') or K2().path.data_path('conf', True)

        '''
        root_path = 'data'
        user_project_path = self.__find_path()
        folder_list = ['..', root_path, user_project_path]
        user_countrepart = self.__current_countreparts()
        if user_countrepart:
            folder_list.append(user_countrepart)
        user_storages = self.__current_stor()
        if user_storages:
            folder_list.append(user_storages)
        if user_folder:
            user = self.__current_usr()
            # Отримати шлях до папки користувача
            folder_list.append('users')
            user_path = user.user_id
            folder_list.append(user_path)
        if work_path:
            folder_list.append(work_path)
        user_folder_path = os.path.join(*folder_list)
        # Перевірити існування структури папок
        if not os.path.exists(user_folder_path):
            # Якщо не існує, створити структуру папок
            os.makedirs(user_folder_path)
        return user_folder_path


    def __find_file_in_path(self, directory, file_name):
        '''Recursive function to find a file in the directory.

            Args:
                directory (str): The absolute path of the directory to search in.
                file_name (str): The name of the file to search for.

            Returns:
                str or None: The absolute path of the file if found, otherwise None.

            Note:
                This function recursively searches for the specified file in the given directory
                and its subdirectories. It gradually reduces the directory path, including more
                parent folders each time, until it finds the file or reaches the root directory.'''


        # Розділити абсолютний шлях на всі його складові
        import platform
        if platform.system() == 'Windows':
            directory = directory.replace("/", '\\')
        path_components = directory.split(os.path.sep)


        # Поступово зменшуємо шлях, включаючи більше верхніх папок кожен раз
        for i in range(len(path_components), 0, -1):
            current_path = os.path.join(*path_components[:i])
            file_path = os.path.join(current_path, file_name)

            # Перевірка, чи файл із зазначеним ім'ям знаходиться в поточній директорії
            if os.path.isfile(file_path):
                return file_path

        # Якщо файл не знайдено в жодній з піддиректорій, повертається значення None
        return None


    def find_data_file(self, file_name):
        '''
        Search for a file in the project hierarchy structure

        Args:
        file_name - str (name of the file to be found)

        Returns: str file path

        Usage: K2().path.find_data_file('data.txt') or K2().path.find_data_file('reports/report_exp.mrt')
        '''


        search_path = []
        if  self.__current_stor():
            search_path.append('../data/' + self.__find_path() + '/' + self.__current_countreparts() + '/' + self.__current_stor() + '/users/' + self.__current_usr().user_id)

        if self.__current_countreparts():
            search_path.append('../data/' + self.__find_path() + '/' + self.__current_countreparts() + '/users/' + self.__current_usr().user_id)

        search_path.append('../data/' + self.__find_path() + '/users/' + self.__current_usr().user_id)
        search_path.append('../data/' + self.__find_path())
        for path in search_path:
            file_path = self.__find_file_in_path(path, file_name)
            if file_path:
                return file_path

        # Якщо файл не знайдено в жодному з шляхів, повертаємо None
        return None

    def media_path(self, component_path, create_date):
        '''
       Create folder hierarchy and return path

       Args:
       component_path - str (end work_path)


       Returns: str media folder path hierarchy

       Usage: K2().path.media_path('crm')

       '''
        root_path = 'media'
        user_project_path = self.__find_path()
        # now = datetime.now()

        year = str(create_date.year)
        month = str(create_date.month).zfill(2)
        day = str(create_date.day).zfill(2)
        #folder list
        folder_list = ['..', root_path, user_project_path, component_path, year, month, day]
        user_folder_path = os.path.join(*folder_list)
        # if exist
        if not os.path.exists(user_folder_path):
            # create if not exist
            os.makedirs(user_folder_path)
        return user_folder_path


    def save_media_file(self, create_date, file, component_name=None):
        '''
        Save the uploaded file to the media path.

        Args:
        create_date: create date of documents or rows
        file: FileStorage - the uploaded file to be saved.
        component_name: the name of the module from which the file was uploaded, if not provided, obtained from the request.

        Returns:
        dict: {'status': True, 'file_path': file_path}
        status True if the file is successfully saved, otherwise False.
        file_path: path to the saved file
        '''

        # Отримуємо розширення файлу
        file_name=os.path.splitext(file.filename)[0]
        file_extension = os.path.splitext(file.filename)[1]
        # Генеруємо унікальне ім'я для файла для запису в БД
        new_filename = self.__generate_id()
        from .k2cfg import K2
        # Якщо ім'я компоненти не передається в аргументах отримуємо його з запиту
        if component_name == None:
            referrer_url = request.referrer
            match = re.match(r'^.*?\/(\w+)\/.*$', referrer_url)
            if match:
                blueprint_name = match.group(1)
                components_data = K2.ins_search_comp()
                if components_data is None or 'components' not in components_data:
                    components_d = []
                else:
                    components_d = components_data['components']
                components = [component['name'] for component in components_d if component.get('installed', True)]

                if blueprint_name in components:
                    component_name = blueprint_name
            else:
                component_name = 'k2root'

        # Повний шлях до файлу
        folder_path = self.media_path(component_name, create_date)  #формування шляху до файлу
        file_path = os.path.join(folder_path, new_filename)
        try:
            # Збереження файлу
            file.save(file_path)
            return {'status': True, 'file_path': file_path, 'file_name': file.filename,  'component_name': component_name} # Файл успішно збережено
        except Exception as e:
            print(f"Error saving file: {e}")
            return {'status': False, 'file_path': None}  # Помилка збереження файлу


    def get_media_file(self, create_date,  file_path, filename, as_attachment=False):
        '''
            Retrieve the path to the file.

            Args:
            create_date: str - createdate of documents or rows
            filepath: str - the unique identifier of the file path.
            filename: str, optional -  name of the file, if not provided, obtained fileid.

            Returns:
            dict or None: {'filedirectory': file_path_result, 'file_path': file_path, 'file_name': filename} if the file is found, otherwise None.
            filedirectory: str - the directory path to the file.
            filepath: str - the hash of the file
            Example:
                K2().path.get_media_file('6hgy3geye3gyyeg3eg', 'my_file')
            '''

        try:
            from .k2cfg import K2
            projid = K2().get_user_project_id()
            project = self.__current_proj(projid)
            file_path_irerach = self.__find_path(project)
            year, month, day = create_date.split('-')

            for comp_name in K2.search_comp_names():
                file_path_result = os.path.join('../media',
                                                file_path_irerach,
                                                comp_name,
                                                year,
                                                month,
                                                day,
                                                file_path)
                if os.path.exists(file_path_result):
                    filedirectory = os.path.join('../media',
                                                file_path_irerach,
                                                comp_name,
                                                year,
                                                month,
                                                day)
                    return send_from_directory(filedirectory, file_path, download_name=filename, as_attachment=as_attachment)
            return None
        except Exception as e:
            logging.error(f'get_media_file {e}')

    def get_media_file_path(self, create_date: str, file_path: str) -> str | None:
        """
        Повертає повний шлях до файлу або None, якщо не знайдено.
        Очікує create_date у форматі 'YYYY-MM-DD' (напр., '2025-10-12').
        """
        try:
            from .k2cfg import K2
            projid = K2().get_user_project_id()
            project = self.__current_proj(projid)
            file_path_irerach = self.__find_path(project)

            # очікується YYYY-MM-DD
            year, month, day = create_date.split('-')
            # базовий корінь медіа
            base = os.path.join('..', 'media', file_path_irerach)

            for comp_name in K2.search_comp_names():
                candidate_dir = os.path.join(base, comp_name, year, month, day)
                candidate = os.path.join(candidate_dir, file_path)
                if os.path.exists(candidate):
                    return os.path.abspath(candidate)  # повертаємо повний шлях

            return None
        except Exception as e:
            logging.error(f'get_media_file_path {e}')
            return None

    def delete_media_file(self, create_date, file_path, filename):
        '''
            Delete the file.

            Args:
            create_date: str - createdate of documents or rows
            filepath: str - the unique identifier of the file path.
            filename: str, optional -  name of the file, if not provided, obtained fileid.

            Returns:
            dict or None: {'filedirectory': file_path_result, 'file_path': file_path, 'file_name': filename} if the file is found, otherwise None.
            filedirectory: str - the directory path to the file.
            filepath: str - the hash of the file
            Example:
                K2().path.delete_media_file('6hgy3geye3gyyeg3eg', 'my_file')
            '''

        try:
            from .k2cfg import K2
            projid = K2().get_user_project_id()
            project = self.__current_proj(projid)
            file_path_irerach = self.__find_path(project)
            year, month, day = create_date.split('/')

            for comp_name in K2.search_comp_names():
                file_path_result = os.path.join('../media',
                                                file_path_irerach,
                                                comp_name,
                                                year,
                                                month,
                                                day,
                                                file_path)
                if os.path.exists(file_path_result):
                    os.remove(file_path_result)
                    return True
            return False
        except Exception as e:
            logging.error(f'delete_media_file {e}')
            return False