# -*- coding: utf-8 -*-
"""
Завантажувач архіву компоненти на сервер оновлення
Запуск: python k2update_push.py

Налаштування (файл розміщено у теці k2/):
  ../builder/config/ignore/<component_name>.txt - список виключень (glob, по одному на рядок)
  ../builder/config/components-list.txt         - список компонент (рядки типу: /k2root або /components/<name>)
  ../builder/config/token.txt                   - JWT токен

Правила:
  - k2root: пакуємо ВМІСТ теки k2/ (де лежить цей скрипт), виключаємо підтеки 'migrations' та 'components'.
  - інші: пакуємо з k2/components/<component_name> через staging-папку (як у первинному файлі).
"""

import os
import json
import fnmatch
import shutil
import zipfile
import requests

from k2.k2cfg import K2
from k2fast_test import fast_tests


# -------------------- Утиліти -------------------- #

def _read_lines(path):
    """Зчитати непорожні рядки (без коментарів #...) з файлу."""
    items = []
    if os.path.exists(path):
        with open(path, 'r', encoding='utf-8') as f:
            for line in f:
                s = line.strip()
                if s and not s.startswith('#'):
                    items.append(s)
    return items


def _should_exclude(rel_path, patterns):
    """Чи відповідає шлях (відносно кореня архіву) будь-якому з glob-патернів."""
    norm = rel_path.replace('\\', '/')
    for p in patterns or ():
        if fnmatch.fnmatch(norm, p) or fnmatch.fnmatch(os.path.basename(norm), p):
            return True
    return False


def _zip_dir(source_root, zip_filename, extra_excludes, root_level_excluded_dirs=None):
    """
    Архівує каталог source_root, виключаючи:
      - кореневі підкаталоги з root_level_excluded_dirs (тільки на першому рівні; опціонально)
      - будь-які шляхи, що співпадають з extra_excludes (із файлу ігнору).
    Вміст у архіві зберігається відносно source_root (без включення верхньої теки).
    """
    root_level_excluded_dirs = set(root_level_excluded_dirs or [])
    os.makedirs(os.path.dirname(zip_filename), exist_ok=True)

    with zipfile.ZipFile(zip_filename, 'w', compression=zipfile.ZIP_DEFLATED) as zf:
        for root, dirs, files in os.walk(source_root):
            rel_root = os.path.relpath(root, source_root)

            # Відсікаємо лише кореневі каталоги (перший рівень)
            if rel_root == '.':
                if root_level_excluded_dirs:
                    dirs[:] = [d for d in dirs if d not in root_level_excluded_dirs]

            # Додаткові виключення директорій за патернами
            dirs[:] = [
                d for d in dirs
                if not _should_exclude(
                    (d if rel_root == '.' else f"{rel_root}/{d}"),
                    extra_excludes
                )
            ]

            # Додаємо файли з урахуванням виключень
            for fname in files:
                rel_path = fname if rel_root == '.' else f"{rel_root}/{fname}"
                if _should_exclude(rel_path, extra_excludes):
                    continue
                abs_path = os.path.join(root, fname)
                zf.write(abs_path, arcname=rel_path)


def _run_tests(comp_name: str, k2_dir: str) -> bool:
    """
    Запускає fast_tests для компонента. Для k2root бігає тести з каталогу k2/.
    Повертає True/False (пройшли/ні).
    """
    if comp_name == 'k2root':
        # 1) Спроба з явним параметром (якщо підтримується)
        try:
            return bool(fast_tests(comp_name, tests_dir=k2_dir))  # type: ignore
        except TypeError:
            pass
        # 2) Старий інтерфейс — змінимо CWD
        cwd = os.getcwd()
        try:
            os.chdir(k2_dir)
            return bool(fast_tests(comp_name))
        finally:
            os.chdir(cwd)
    else:
        return bool(fast_tests(comp_name))


# -------------------- Основна логіка -------------------- #

def update_push():
    # k2_dir = каталог цього файлу (скрипт лежить у k2/)
    k2_dir = os.path.abspath(os.path.dirname(__file__))

    # === Конфіги (відносно k2/) ===
    token_file = os.path.normpath(os.path.join(k2_dir, '..', 'builder', 'config', 'token.txt'))
    components_list_file = os.path.normpath(os.path.join(k2_dir, '..', 'builder', 'config', 'components-list.txt'))

    # === Токен ===
    with open(token_file, 'r', encoding='utf-8') as file:
        jwt_token = file.read().strip()

    # === URL сервера оновлень ===
    server_url = K2.update_domain + '/k2update/api/save-archive-components-with-tokens/'
    # server_url = 'http://127.0.0.1:7654/k2update/api/save-archive-components-with-tokens/'

    # === Список компонентів ===
    raw_components = _read_lines(components_list_file)
    if not raw_components:
        print("Список компонент порожній або не знайдений.")
        return

    # Папка для артефактів (../builder/components)
    components_out_dir = os.path.normpath(os.path.join(k2_dir, '..', 'builder', 'components'))
    os.makedirs(components_out_dir, exist_ok=True)

    # === Цикл по компонентах ===
    for comp_entry in raw_components:
        if not comp_entry:
            continue
        # Допускаємо записи типу "/k2root" або "/components/<name>"
        rel = comp_entry.strip().lstrip('/')
        comp_name = rel.split('/')[-1] if rel else rel

        try:
            # Ignore-патерни для конкретного компонента
            ignore_file = os.path.normpath(os.path.join(k2_dir, '..', 'builder', 'config', 'ignore', f'{comp_name}.txt'))
            ignore_patterns = _read_lines(ignore_file)

            # Куди покладемо zip та staging (як у первинному варіанті)
            zip_filename = os.path.join(components_out_dir, f'{comp_name}.zip')
            staging_dir = os.path.join(components_out_dir, comp_name)

            if rel == 'k2root':
                # ЛОГІКА НЕ ЧІПАЄМО: пакуємо вміст k2/, ігноруємо k2/migrations та k2/components
                _zip_dir(
                    source_root=k2_dir,
                    zip_filename=zip_filename,
                    extra_excludes=ignore_patterns,
                    root_level_excluded_dirs={'migrations', 'components', 'auto_update', '.pytest_cache'}
                )

            else:
                # БЕРЕМО ЛИШЕ з k2/components/<component_name>
                comp_abs_path = os.path.join(k2_dir, 'components', comp_name)
                if not os.path.exists(comp_abs_path):
                    raise FileNotFoundError(f"Компонент не знайдено: {comp_abs_path}")

                # Створюємо staging-папку, як у первинному файлі
                os.makedirs(staging_dir, exist_ok=True)

                # Копіюємо компонент всередину staging/<name> з ігнорами
                shutil.copytree(
                    comp_abs_path,
                    os.path.join(staging_dir, comp_name),
                    ignore=shutil.ignore_patterns(*ignore_patterns) if ignore_patterns else None,
                    dirs_exist_ok=True
                )

                # Створюємо архів ЗІ staging-папки (ВАЖЛИВО: як у первинному файлі!)
                # Це створить файл "<staging_dir>.zip" — тобто рівно '../builder/components/<name>.zip'
                shutil.make_archive(staging_dir, 'zip', staging_dir)

                # Прибираємо staging
                shutil.rmtree(staging_dir, ignore_errors=True)

            # --- ТЕСТИ ---
            if comp_name != "k2root":
                # --- ТЕСТИ ---
                is_passed = _run_tests(comp_name, k2_dir)
                if not is_passed:
                    print("Тести не пройшли, архів не відправляється")
                    return

        except Exception as e:
            print(f"[{comp_entry}] Помилка підготовки архіву: {e}")
            continue

        # === Надсилання архіву на сервер ===
        headers = {'Authorization': f'Bearer {jwt_token}'}
        file_name_on_server = f'{comp_name}.zip'

        try:
            with open(zip_filename, 'rb') as fh:
                files = {'file': (file_name_on_server, fh)}
                response = requests.post(server_url, files=files, headers=headers)
        except Exception as e:
            print(f"[{comp_entry}] Помилка відправки: {e}")
            continue

        if response.status_code == 200:
            try:
                print(response.json())
            except json.decoder.JSONDecodeError:
                print("[WARN] Відповідь 200, але не JSON.")
        else:
            print(f"[{comp_entry}] Помилка відповіді сервера: {response.status_code}")
            try:
                print(response.json())
            except Exception:
                print(response.text)


if __name__ == "__main__":
    update_push()
