import sys
import logging
import requests
from k2.k2cfg import K2

FLASK_API_URL = f"{K2.domain}k2pubsub/api/pubsub-receive"
streaming_pull_future = None

def read_pubsub_messages(project_id: str, subscription_id: str, service_account_file: str):
    from google.cloud import pubsub_v1
    import json

    # Initialize the Pub/Sub client
    subscriber = pubsub_v1.SubscriberClient.from_service_account_file(service_account_file)

    # Define the subscription path
    subscription_path = subscriber.subscription_path(project_id, subscription_id)

    def callback(message):
        try:
            data = message.data.decode("utf-8")
            json_data = json.loads(data)  # Перетворюємо у JSON
            logging.info(f"Received message: {json_data}")

            # Відправляємо отримані дані у Flask
            response = requests.post(FLASK_API_URL, json={
                "subscription_id": subscription_id,
                "message": json_data
            })
            logging.info(f"Sent to Flask, response: {response.status_code}, {response.text}")

            # Якщо успішно передали у Flask, підтверджуємо отримання
            message.ack()
        except Exception as e:
            logging.error(f"Error processing message: {e}")

    # Subscribe to the specified subscription and start receiving messages
    streaming_pull_future = subscriber.subscribe(subscription_path, callback=callback)

    print(f"Listening for messages on {subscription_path}...\n")

    # Keep the script running to continue receiving messages
    try:
        streaming_pull_future.result()
    except KeyboardInterrupt:
        streaming_pull_future.cancel()
        print("Subscription cancelled.")

if __name__ == "__main__":
    if len(sys.argv) < 4:
        print("Usage: python k2pubsub_service_read.py <project_id> <subscription_id> <service_account_file>")
        sys.exit(1)

    project_id = sys.argv[1]
    subscription_id = sys.argv[2]
    service_account_file = sys.argv[3]

    read_pubsub_messages(project_id, subscription_id, service_account_file)