#!/usr/bin/env bash
set -euo pipefail

# Визначаємо PROJECT_ROOT = шлях до K2CloudERP/
CWD="$(pwd)"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

if [ "$(basename "$CWD")" = "K2CloudERP" ]; then
  PROJECT_ROOT="$CWD"
elif [ -d "$CWD/K2CloudERP" ]; then
  PROJECT_ROOT="$CWD/K2CloudERP"
elif [ "$(basename "$SCRIPT_DIR")" = "K2CloudERP" ]; then
  PROJECT_ROOT="$SCRIPT_DIR"
else
  echo "Не вдалося знайти каталог проекту K2CloudERP."
  echo "Запускайте цей скрипт зі всередини K2CloudERP/ або з батьківської директорії, що містить K2CloudERP/."
  exit 1
fi

VENV_DIR="${PROJECT_ROOT%/}/venv"

# Переконаємось, що є python3.12
PYBIN="$(command -v python3.12 || true)"
if [ -z "${PYBIN}" ]; then
  echo "python3.12 не знайдено. Спочатку запустіть: bash check_python.sh (у K2CloudERP/ або вище)."
  exit 1
fi

# Якщо існує venv, але без pip — перевідтворюємо начисто
if [ -d "$VENV_DIR" ] && [ ! -x "$VENV_DIR/bin/pip" ]; then
  echo "[INFO] Існуючий venv без pip — перевідтворюю..."
  rm -rf "$VENV_DIR"
fi

if [ ! -d "$VENV_DIR" ]; then
  echo "[INFO] Створюю venv на Python 3.12: $VENV_DIR"

  # 1) Базове створення venv без апдейту залежностей
  "$PYBIN" -m venv "$VENV_DIR" || "$PYBIN" -m venv --without-pip "$VENV_DIR"

  # 2) Спроба підняти pip через ensurepip усередині venv
  if ! "$VENV_DIR/bin/python" -m ensurepip --upgrade --default-pip; then
    echo "[WARN] ensurepip у venv впав. Пробую альтернативу через virtualenv…"
    if ! command -v virtualenv >/dev/null 2>&1; then
      # спробуємо доставити virtualenv (на apt-системі пакет python3-virtualenv)
      if command -v apt-get >/dev/null 2>&1; then
        sudo apt-get -o Dir::Etc::sourceparts=/dev/null -o Dir::Etc::sourcelist=/etc/apt/sources.list update -y || true
        sudo apt-get -o Dir::Etc::sourceparts=/dev/null -o Dir::Etc::sourcelist=/etc/apt/sources.list install -y python3-virtualenv || true
      fi
    fi
    if command -v virtualenv >/dev/null 2>&1; then
      rm -rf "$VENV_DIR"
      virtualenv -p "$PYBIN" "$VENV_DIR"
    else
      echo "[ERR] Немає working ensurepip і немає virtualenv. Переконайтеся, що встановлено python3.12-venv або python3-virtualenv."
      exit 1
    fi
  fi
  echo "[OK] venv створено: $VENV_DIR"
else
  echo "[INFO] venv уже існує: $VENV_DIR"
fi

# 3) Оновимо інструменти в будь-якому разі
if [ -x "$VENV_DIR/bin/pip" ]; then
  "$VENV_DIR/bin/python" -m pip install --upgrade pip setuptools wheel || true
else
  echo "[ERR] Після створення venv pip відсутній: $VENV_DIR/bin/pip"
  echo "     Перевстановіть пакет python3.12-venv (або python3-virtualenv) і повторіть."
  exit 1
fi

# Фінальна перевірка
if ! "$VENV_DIR/bin/python" -V 2>/dev/null | grep -q "Python 3\.12"; then
  echo "[WARN] venv не на Python 3.12. Поточний: $("$VENV_DIR/bin/python" -V 2>/dev/null || echo 'невідомо')"
else
  echo "[OK] venv готовий: $("$VENV_DIR/bin/python" -V)"
fi

