# app.py

# -------------------- gevent / psycogreen патчі --------------------
from gevent import monkey
monkey.patch_all(thread=False)

import logging
import os
from datetime import timedelta, datetime

import psycopg2, psycogreen.gevent  # noqa: F401
from psycopg2 import extensions
from psycogreen.gevent import patch_psycopg
patch_psycopg()
print("wait_cb:", extensions.get_wait_callback())

# -------------------- Flask eco --------------------
from flask import Flask, request, g, has_request_context, session, request, current_app
from flask_cors import CORS
from flask_wtf import CSRFProtect
from flask_session import Session
from flask_login import current_user, LoginManager, login_required
from flask_socketio import SocketIO
from ua_parser import user_agent_parser
import psutil
import secrets
# from flask_debugtoolbar import DebugToolbarExtension  # опційно

# -------------------- SQLAlchemy / міграції --------------------
from sqlalchemy import event
from sqlalchemy.engine import Engine
from flask_sqlalchemy import SQLAlchemy
from flask_migrate import Migrate

# -------------------- Swagger --------------------
from flasgger import Swagger

# -------------------- K2 інфраструктура --------------------
from k2.k2logging import configure_logging
from k2.k2trans import K2Babel
from k2.k2cfg import K2
from k2.k2obj import K2Obj
from k2.k2monitoring import Monitoring


# -------------------- Flask app --------------------
class K2Flask(Flask):
    def _check_setup_finished(self, f_name: str) -> None:
        # Приглушуємо внутрішню перевірку, бо ініціалізуємо поступово
        pass


app = K2Flask(__name__)
CORS(app)

# Базові налаштування
app.config['SECRET_KEY'] = K2.secret_key
app.json.sort_keys = K2.json_sort_keys
app.config['WTF_CSRF_CHECK_DEFAULT'] = False
csrf = CSRFProtect(app)

# -------------------- Logging --------------------
configure_logging()

# check redis init
try:
    message_queue = None
    from k2.k2clients_connection import redis_client
    if redis_client:
        message_queue = K2().redis_uri_string()
    else:
        message_queue = None
except Exception as e:
    logging.error("Error import RedisInstance")


# Socket.IO (gevent)
socketio = SocketIO(
    app,
    async_mode='gevent',
    cors_allowed_origins="*",
    manage_session=True,
    message_queue= message_queue, #message_queue,
    ping_timeout=180
)

# Сесії
app.config['SESSION_TYPE'] = 'filesystem'
app.config['SESSION_FILE_DIR'] = '../sessions'
Session(app)

# ENV (корисно для підпроцесів / CLI)
pythonpath = os.environ.get('PYTHONPATH', '')
pythonpath += ':.'
os.environ['PYTHONPATH'] = pythonpath
os.environ['FLASK_APP'] = 'app.py'

# -------------------- Monitoring --------------------
monitor = Monitoring(
    app,
    window_size=10_000,
    slow_query_threshold=0.5,
    slow_route_threshold=2.0,
    register_endpoint=True,
    endpoint_url="/metrics",
    filter_static=True,
    sample_rate=1.0,
    require_login=True,
)

# -------------------- DB / SQLAlchemy --------------------
app.config['SQLALCHEMY_DATABASE_URI'] = K2().init_db()
app.config['SQLALCHEMY_POOL_SIZE'] = K2.sglalchemy_pool_size
app.config['SQLALCHEMY_MAX_OVERFLOW'] = K2.sglalchemy_max_overflow
app.config['SQLALCHEMY_ENGINE_OPTIONS'] = {
    'pool_size': K2.sglalchemy_pool_size,
    'max_overflow': K2.sglalchemy_max_overflow,
}
app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False  # без зайвого оверхеду

app.config['REMEMBER_COOKIE_DURATION'] = timedelta(days=21)
app.config['PERMANENT_SESSION_LIFETIME'] = timedelta(minutes=180)

# Jinja: приємні дрібниці
app.jinja_env.trim_blocks = True
app.jinja_env.lstrip_blocks = True

# Ініт БД
db = SQLAlchemy()
db.init_app(app)
K2.db = db

with app.app_context():
    # Бери engine за замовчуванням
    try:
        default_engine = getattr(db, "engines", {}).get(None, db.engine)
    except Exception:
        default_engine = db.engine

    # Потрібно деяким класам, які очікують "голий" Engine
    K2Obj.db = default_engine
    # Передача екземпляра об'єкта SQLAlchemy у базовий клас K2Obj
    K2Obj.database = db
    # Підключення кастомних engine (разово, не на кожен запит)
    K2().db_custom_engine(db)

    # Моніторинг SQL
    try:
        monitor.attach_sqlalchemy(default_engine)
    except Exception:
        pass




# -------------------- Migrations --------------------
migrate = Migrate(app, db, autogenerate=True)

# -------------------- Translations --------------------
K2.load_babel_translation_directories()
app.config['BABEL_TRANSLATION_DIRECTORIES'] = K2.babel_translation_directories
app.config['BABEL_DEFAULT_LOCALE'] = K2.current_language or K2.default_language
app.config['BABEL_LANGUAGES'] = K2.languages
app.config['BABEL_DEFAULT_TIMEZONE'] = K2.timezone
babel = K2Babel(app)
babel.init_app(app, locale_selector=K2.get_locale)

# Список компонентів (кешується на класі)
K2.component_list()

# -------------------- Swagger (login required) --------------------
swagger_config = {
    "headers": [],
    "specs": [
        {
            "endpoint": "route-info",
            "route": "/route-info-spec",
            "rule_filter": lambda rule: True,
            "model_filter": lambda tag: True,
        }
    ],
    "swagger_ui": False,
}
swagger = Swagger(app, config=swagger_config, parse=True)


def protect_route_info_spec():
    func = app.view_functions.get('flasgger.route-info')
    if func:
        app.view_functions['flasgger.route-info'] = login_required(func)
    else:
        print("Route /route-info-spec (flasgger.route-info) not found!")
protect_route_info_spec()


# -------------------- Request lifecycle --------------------
@app.before_request
def before_request():
    # user-agent (розбір один раз)
    g.user_agent = user_agent_parser.Parse(request.headers.get('User-Agent', ''))
    # один екземпляр K2 на запит
    if not hasattr(g, "k2"):
        g.k2 = K2()
    # домен для поточного запиту
    K2.domain = f"{K2.domain_protocol}{request.host}/"
    # для нестатичних маршрутів
    if request.endpoint and 'static' not in str(request.endpoint):
        g.start_time = datetime.now()
        g.k2.page_permission = g.k2.get_user_permissions()
        K2.page_permission = g.k2.page_permission

        # контроль виходу користувача адміністратором
        # try:
        #     if current_user.is_authenticated:
        #         g.k2.check_logout_users(current_user.get_id())
        # except Exception:
        #     pass

        # # режим db_login: перемикаємо Engine під користувача (не роби цього щоразу без потреби)
        # if K2.db_login:
        #     try:
        #         g.k2.db_user_engine(db)
        #     except Exception:
        #         pass


# @app.before_request
# def presence_http_beat():
#     try:
#         if not current_user.is_authenticated:
#             return
#
#         from components.k2site.k2site.objects.presence.presence_service import heartbeat, TTL_SECONDS
#         sid = session.get('presence_sid')
#         if not sid:
#             sid = secrets.token_hex(8)  # стабільний для цієї сесії браузера
#             session['presence_sid'] = sid
#         heartbeat(
#             str(current_user.get_id()),
#             sid,
#             {"path": request.path, "room": request.args.get("room"), "idle": False},
#             ttl=TTL_SECONDS
#         )
#     except Exception as e:
#         logging.error(str(e))

@app.errorhandler(404)
def page_not_found(error):
    return "404 - Page not found!", 404


@app.context_processor
def inject_static_variable():
    """
    K2 кешує контекст користувача в g._k2_ctx (один JOIN-запит на запит).
    """
    if hasattr(g, "k2"):
        k2 = g.k2
    else:
        k2 = K2()

    g.user_agent = user_agent_parser.Parse(request.headers.get('User-Agent', ''))

    # Меню / категорії / словник класів
    k2.search_menu_items()
    if not K2.menu_category:
        k2.search_menu_items_category()
    if not K2.class_dict:
        k2.search_class_dict()
    k2.get_menu_url()
    g.all_routes = k2.menu_url

    # Помилки для сторінки
    try:
        error_dict = K2.load_logging_messages(getattr(request, "url_rule", None))
        error_count = error_dict.get('error_count', 0)
        warning_count = error_dict.get('warning_count', 0)
        error_messages = error_dict.get('error_messages', [])
    except Exception:
        error_count = 0
        warning_count = 0
        error_messages = []

    # Нотифікації — лише для залогіненого
    try:
        notifications = k2.notifications.get_user_notifications() if current_user.is_authenticated else []
    except Exception:
        notifications = []

    context = {
        'menu': K2.menu,
        'menu_category': K2.menu_category,

        'template_name': k2.current_template(),
        'current_user_login': k2.current_user_login,
        'current_user_role': K2.get_current_user_role_name(),
        'current_user_permissions': getattr(k2, "page_permission", {}) if k2 else {},

        'locale': K2.get_locale(),
        'request': request,
        'user_agent': g.user_agent,
        'count_update': 0,
        'port': K2.port,
        'proj': K2.proj_config,

        # Імена/ідентифікатори без додаткових SELECT (беруться з кешу в g._k2_ctx)
        'countreparts': k2.get_user_counterparts_name(),
        'storages': k2.get_user_stoages_name(),
        'project_name': k2.get_user_project_name(),

        'end_time': datetime.now(),
        'domain': K2.domain,
        'domain_comp': '',
        'online_users': k2.get_authorized_users(),
        'page_id': K2.namemenu(request.path),
        'page_hash': K2.generate_id(),

        'error_count': error_count,
        'warning_count': warning_count,
        'error_messages': error_messages,

        'system_settings': K2.system_settings,
        'key_syncfusion': K2.key_syncfusion,
        'key_aggrid': K2.key_aggrid,
        'key_stimulsoft': K2.key_stimulsoft,
        'notifications': notifications,
        'current_language': K2.get_locale(),
        'version': k2.version,
    }

    context['domain_comp'] = f"{K2.domain}/components" if K2.nginx_static else K2.domain
    return context


# Jinja bytecode cache (за потреби)
if K2.system_settings and K2.system_settings.get('caching_enabled'):
    from jinja2 import FileSystemBytecodeCache
    cache_dir = '../cache'
    os.makedirs(cache_dir, exist_ok=True)
    app.jinja_env.bytecode_cache = FileSystemBytecodeCache(directory=cache_dir)


@app.after_request
def after_request(response):
    try:
        endpoint = request.endpoint
        if not endpoint or 'static' in str(endpoint):
            return response

        # Лог відвідування
        from components.k2site.k2site.views import K2Site
        # print(g.__dict__)
        # k2 = g.k2
        should_log = False
        if current_user.is_authenticated:
            should_log = (request.path in getattr(g, 'all_routes', set())) or \
                         (request.args.get('parent') in getattr(g, 'all_routes', set())) or \
                         ('api' in request.path)
        else:
            should_log = True  # дозволити логувати гостей за потреби

        if should_log:
            try:
                K2Site().user_login_save_log(
                    current_user,
                    g.user_agent,
                    request.headers.get('X-Forwarded-For', request.remote_addr),
                    'Open page',
                    request.path,
                    psutil.cpu_percent(interval=0),
                    psutil.virtual_memory().percent,
                    psutil.disk_usage('/').percent,
                    request.headers.get('Authorization')
                )
            except Exception as e:
                logging.error(f"After reqest error {e}")
    except Exception as e:
        logging.error(f"After reqest error {e}")
    return response


# # ------------------ підрахунок к-ті SQL -----------------------------

# @app.before_request
# def _init_sql_counter():
#     g.sql_queries = 0
#
# @event.listens_for(Engine, "before_cursor_execute")
# def _count_global(conn, cursor, statement, parameters, context, executemany):
#     if hasattr(g, "sql_queries"):
#         sql = (statement or "").strip().lower()
#         # Ігноруємо службові SET-и аудиту, щоб не «роздувати» рахунок
#         if sql.startswith("set local myapp.user_"):
#             return
#         g.sql_queries += 1
#
#
# @app.after_request
# def _add_header_sql_count(response):
#     if hasattr(g, "sql_queries"):
#         response.headers["X-SQL-Queries"] = str(g.sql_queries)
#         if g.sql_queries >= 50:
#             app.logger.warning("HIGH SQL COUNT %s -> %s", request.path, g.sql_queries)
#     return response


# Корисно для локальної діагностики SQL
# logging.getLogger("sqlalchemy.engine").setLevel(logging.INFO)


#-------------------- SQLAlchemy listener: аудит через SET LOCAL --------------------
if has_request_context() and current_user.is_authenticated:
    @event.listens_for(Engine, "before_cursor_execute")
    def set_user_for_audit(conn, cursor, statement, parameters, context, executemany):
        """
        Встановлює ID та логін поточного користувача в тимчасові змінні сесії PostgreSQL.
        """
        try:
            # Перевіряємо, що є залогінений користувач
            if current_user and current_user.is_authenticated:
                # 1. Передаємо ID користувача
                user_id = str(current_user.get_id())
                cursor.execute("SET LOCAL myapp.user_id = %s", (user_id,))

                # 2. Передаємо логін користувача
                login = None
                if hasattr(current_user, 'get_login'):
                    login = str(current_user.get_login())

                if login:
                    cursor.execute("SET LOCAL myapp.user_login = %s", (login,))
            pass
        except Exception as e:
            # Ігноруємо помилки, якщо код виконується поза контекстом
            # веб-запиту (напр. при старті), щоб не "зламати" додаток.
            pass



# @event.listens_for(Engine, "before_cursor_execute")
# def set_user_for_audit(conn, cursor, statement, parameters, context, executemany):
#     try:
#         if not has_request_context():
#             return
#         ctx = getattr(g, "_k2_ctx", None)
#         user_id = None
#         if ctx:
#             user_id = getattr(ctx, 'user_id', None)
#             login = getattr(g, "login", None)
#         if not user_id:
#             return
#
#         is_named = bool(getattr(cursor, "name", None))
#         if is_named:
#             tmp = conn.cursor()
#             try:
#                 tmp.execute("SET LOCAL myapp.user_id = %s", (str(user_id),))
#                 if login:
#                     tmp.execute("SET LOCAL myapp.user_login = %s", (str(login),))
#             finally:
#                 try: tmp.close()
#                 except Exception: pass
#         else:
#             cursor.execute("SET LOCAL myapp.user_id = %s", (str(user_id),))
#             if login:
#                 cursor.execute("SET LOCAL myapp.user_login = %s", (str(login),))
#     except Exception as e:
#         logging.info(e)


@app.teardown_appcontext
def shutdown_session(exception=None):
    db.session.remove()


# Платформа (для інфо)
K2.platform = K2.get_platform()

# Маршрути (Blueprints/Views)
from routes import *  # noqa: E402,F401


if __name__ == '__main__':
    socketio.run(app, port=int(K2.port))
