import functools
import inspect
import json
import logging

import redis
from cachetools import TTLCache
from k2.k2cfg import K2


class CacheManager:
    # def __init__(self, host="localhost", port=6379, password="k2adm123456", db=0):
    #     pool = redis.ConnectionPool(
    #         host=host,
    #         port=port,
    #         db=db,
    #         password=password,
    #         decode_responses=True,
    #         max_connections=10,
    #     )
    def __init__(self, redis_client=None):
        if redis_client is not None:
            self.client = redis_client
        else:
            redis_config = K2().init_redis_uri() or {}
            pool = redis.ConnectionPool(
                host=redis_config.get("host", "localhost"),
                port=redis_config.get("port", 6379),
                db=redis_config.get("db", 0),
                password=redis_config.get("password", "k2adm123456"),
                decode_responses=True,
                max_connections=10,
            )
            self.client = redis.Redis(connection_pool=pool)

        try:
            self.client.ping()
            self.redis_available = True
            logging.info("Redis connection successful")
        except redis.RedisError as e:
            logging.error(f"Redis connection failed: {e}")
            self.redis_available = False
            self.fallback_cache = TTLCache(maxsize=1000, ttl=300)

    def _make_key(self, namespace: str, key: str) -> str:
        """Створює ключ з namespace"""
        return f"{namespace}:{key}"

    def set(self, namespace: str, key: str, value: any, ex: int = None):
        """Універсальний set з namespace"""
        full_key = self._make_key(namespace, key)

        if self.redis_available:
            value_json = json.dumps(value, ensure_ascii=False)
            self.client.set(full_key, value_json, ex=ex)
        else:
            self.fallback_cache[full_key] = value

    def get(self, namespace: str, key: str) -> any:
        """Універсальний get з namespace"""
        full_key = self._make_key(namespace, key)

        if self.redis_available:
            data = self.client.get(full_key)
            return json.loads(data) if data else None
        else:
            return self.fallback_cache.get(full_key)

    def delete(self, namespace: str, key: str) -> bool:
        """Видаляє ключ"""
        full_key = self._make_key(namespace, key)

        if self.redis_available:
            return bool(self.client.delete(full_key))
        else:
            return self.fallback_cache.pop(full_key, None) is not None

    def clear_namespace(self, namespace: str) -> int:
        """Очищає весь namespace"""
        if self.redis_available:
            pattern = f"{namespace}:*"
            keys = self.client.keys(pattern)
            return self.client.delete(*keys) if keys else 0
        else:
            keys_to_delete = [
                k for k in self.fallback_cache.keys() if k.startswith(f"{namespace}:")
            ]
            for key in keys_to_delete:
                del self.fallback_cache[key]
            return len(keys_to_delete)

    def cached(self, timeout=None, make_key=None, from_cache=None):
        """
        Повністю універсальний декоратор для кешування.
        - `timeout`: час життя кешу в секундах.
        - `make_key`: функція для створення кастомного ключа.
        - `from_cache`: функція для відновлення об'єкта зі словника (напр., Config.from_dict).
        """

        def decorator(func):
            @functools.wraps(func)
            def wrapper(*args, **kwargs):
                # 1. Створення ключа кешу
                if make_key:
                    cache_key = make_key(*args, **kwargs)
                else:
                    args_for_key = args
                    try:
                        params = inspect.signature(func).parameters
                        if params and list(params.keys())[0] in ("self", "cls"):
                            args_for_key = args[1:]
                    except Exception:
                        pass

                    sorted_kwargs = tuple(sorted(kwargs.items()))
                    cache_key = f"func:{func.__name__}:{args_for_key}:{sorted_kwargs}"

                # 2. Перевірка кешу
                cached_result = self.get(cache_key)
                if cached_result is not None:
                    logging.info(
                        f"Decorator CACHE HIT for key: {self.key_prefix}{cache_key}"
                    )
                    if from_cache:
                        return from_cache(cached_result)
                    return cached_result

                logging.info(
                    f"Decorator CACHE MISS for key: {self.key_prefix}{cache_key}"
                )

                # 3. Виклик оригінальної функції
                result = func(*args, **kwargs)

                # 4. Збереження результату в кеш
                result_to_cache = result
                if hasattr(result, "to_dict"):
                    result_to_cache = result.to_dict()

                self.set(cache_key, result_to_cache, ex=timeout)

                return result

            return wrapper

        return decorator
