import os
import subprocess
import sys
import platform


def fast_tests(component_name: str = None):
    """Запускає pytest з правильними шляхами до модулів"""
    project_root = os.path.abspath(os.path.dirname(__file__))
    k2_root = os.path.dirname(project_root)

    # Шлях до віртуального середовища
    operating_system = platform.system()
    venv_path = os.path.dirname(k2_root)
    if operating_system == 'Windows':
        python_executable = os.path.join(venv_path, "venv", "Scripts", "python.exe")
    elif operating_system == 'Linux':
        python_executable = os.path.join(venv_path, "venv", "bin", "python")
    # Перевірка наявності python.exe
    if not os.path.isfile(python_executable):
        print(f"❌ Не знайдено файл: {python_executable}")
        return

    # Список шляхів для додавання в sys.path
    required_paths = [
        k2_root,
        os.path.join(k2_root, "k2tests"),
        os.path.join(k2_root, "components"),
        venv_path
    ]

    # Додаємо шляхи до sys.path, якщо вони ще не додані
    for path in required_paths:
        if path not in sys.path:
            sys.path.insert(0, path)

    # Запит компоненти для запуску тестів
    component = input("\nВведіть назву компоненти (або Enter для виходу): ").strip()
    if not component:
        print("❌ Вихід.")
        return

    # Формуємо шлях до тестів компоненти
    component = os.path.normpath(component).replace(os.path.sep, "_")
    component_test_path = os.path.join("components", component, component, "tests")

    # Перевірка, чи існують тести для обраної компоненти
    if not os.path.isdir(os.path.join(k2_root, component_test_path)):
        print(f"❌ Тести для '{component}' не знайдено!")
        return

    print(f"\nЗапуск тестів для '{component}'...\n")

    # Копіюємо змінні середовища
    env = os.environ.copy()

    # Додаємо потрібні шляхи до PYTHONPATH
    env['PYTHONPATH'] = os.pathsep.join([k2_root] + sys.path)

    # Встановлюємо додаткові змінні середовища
    env['FLASK_APP'] = 'k2.app'

    # Перевірка чи встановлено pytest
    try:
        subprocess.run([python_executable, "-m", "pip", "show", "pytest"], check=True, capture_output=True)
        print("pytest успішно знайдено.")
    except subprocess.CalledProcessError:
        print("❌ Помилка: pytest не встановлено в середовищі.")
        return

    # Запуск тестів через pytest
    try:
        subprocess.run(
            [python_executable, "-m", "pytest", component_test_path, "-v", "-s", "--tb=long", "--disable-warnings"],
            cwd=k2_root,
            check=True,
            env=env
        )
    except subprocess.CalledProcessError as e:
        print(f"❌ Помилка pytest (код {e.returncode})")
    except FileNotFoundError:
        print("❌ Помилка: не знайдено Python або pytest!")


if __name__ == "__main__":
    fast_tests()
