#!/usr/bin/env bash
set -euo pipefail

# ------------------------------------------------------------
# Працюємо з каталогу, де лежить цей скрипт (K2CloudERP/)
# ------------------------------------------------------------
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR"

# ------------------------------------------------------------
# Параметри
# ------------------------------------------------------------
port="7654"
proj="k2"
for arg in "$@"; do
  case $arg in
    --port=*) port="${arg#*=}";;
    --proj=*) proj="${arg#*=}";;
  esac
done

# proj.yml у поточному каталозі (K2CloudERP/)
{
  echo "port: $port"
  echo "proj: $proj"
} > proj.yml

echo "[INFO] Using port=$port proj=$proj"

# ------------------------------------------------------------
# 1) Перевіряємо Python 3.12 + системні dev-залежності (APT Variant 1)
#    (залиште ./ якщо файл у цьому каталозі; якщо зберігаєте у батьківській — поміняйте на ../check_python.sh)
# ------------------------------------------------------------
bash ./check_python.sh

# ------------------------------------------------------------
# 2) Створюємо/ремонтуємо venv у K2CloudERP/venv
# ------------------------------------------------------------
bash ./create_venv.sh

# ------------------------------------------------------------
# 3) Активуємо venv і ставимо Python-залежності
# ------------------------------------------------------------
# shellcheck disable=SC1091
source ./venv/bin/activate

python -m pip install --upgrade pip setuptools wheel

# Спроба поставити готові колеса (без збірки) для найпроблемніших пакетів:
# Якщо якесь колесо недоступне — команда може впасти; дозволяємо продовжити.
pip install --no-cache-dir --only-binary=:all: \
  psycopg2==2.9.10 mysqlclient==2.2.7 || true

# Основні залежності проєкту (без приглушення помилок!)
REQ_FILE="./k2/requirements.txt"

if [ -f "$REQ_FILE" ]; then
  echo "[INFO] Installing $REQ_FILE …"
  python -m pip install --no-cache-dir -r "$REQ_FILE"
else
  echo "[WARN] $REQ_FILE не знайдено"
fi

# Якщо після інсталяції psycopg2 все ще не імпортується — підстрахуємось бінарним пакетом
if ! python - <<'PY'
import importlib.util, sys
sys.exit(0 if importlib.util.find_spec("psycopg2") else 1)
PY
then
  echo "[WARN] psycopg2 не знайдено у venv — ставлю psycopg2-binary як тимчасовий обхід…"
  python -m pip install --no-cache-dir psycopg2-binary==2.9.10
fi

# ------------------------------------------------------------
# 4) Підготовка логів та запуск gunicorn
# ------------------------------------------------------------
LOG_DIR="$SCRIPT_DIR/logs"
mkdir -p "$LOG_DIR"
cd k2

echo "[INFO] Starting gunicorn on 127.0.0.1:${port} …"
exec gunicorn \
  -k geventwebsocket.gunicorn.workers.GeventWebSocketWorker \
  --bind "127.0.0.1:${port}" \
  app:app \
  --graceful-timeout 360 \
  --log-level=debug \
  --access-logfile="$LOG_DIR/access.log"

